#include <avr/io.h>
#include "utils.h"

#ifndef cbi
#define cbi(sfr, bit) (_SFR_BYTE(sfr) &= ~_BV(bit))
#endif
#ifndef sbi
#define sbi(sfr, bit) (_SFR_BYTE(sfr) |= _BV(bit))
#endif






long map(long x, long in_min, long in_max, long out_min, long out_max)
{
  return (x - in_min) * (out_max - out_min) / (in_max - in_min) + out_min;
}//end map


uint32_t constrain(uint32_t value, uint32_t min, uint32_t max)
{
    if(value < min) value = min;
    if(value > max) value = max;
    return value;
}//end constrain





void delayMicroseconds(uint16_t us)
{
    // calling avrlib's delay_us() function with low values (e.g. 1 or
    // 2 microseconds) gives delays longer than desired.
    //delay_us(us);

#if F_CPU >= 16000000L
    // for the 16 MHz clock on most Arduino boards

    // for a one-microsecond delay, simply return.  the overhead
    // of the function call yields a delay of approximately 1 1/8 us.
    if (--us == 0)
        return;

    // the following loop takes a quarter of a microsecond (4 cycles)
    // per iteration, so execute it four times for each microsecond of
    // delay requested.
    us <<= 2;

    // account for the time taken in the preceeding commands.
    us -= 2;
#else
    // for the 8 MHz internal clock on the ATmega168

    // for a one- or two-microsecond delay, simply return.  the overhead of
    // the function calls takes more than two microseconds.  can't just
    // subtract two, since us is unsigned; we'd overflow.
    if (--us == 0)
        return;
    if (--us == 0)
        return;

    // the following loop takes half of a microsecond (4 cycles)
    // per iteration, so execute it twice for each microsecond of
    // delay requested.
    us <<= 1;

    // partially compensate for the time taken by the preceeding commands.
    // we can't subtract any more than this or we'd overflow w/ small delays.
    us--;
#endif

    // busy wait
    __asm__ __volatile__ (
        "1: sbiw %0,1" "\n\t" // 2 cycles
        "brne 1b" : "=w" (us) : "0" (us) // 2 cycles
    );
}//end delayMicroseconds








void delay(uint16_t milliseconds)
{
    for(uint16_t i = 0; i < milliseconds; i++)
    {
        delayMicroseconds(1000);
    }
}//end delay






//this is modified to accept portB pin numbers,
//instead of ADC channel number like it used to
int analogRead(uint8_t pinPortB)
{

    uint8_t low, high, adcChannelNumber;

    //convert port B pin number to ADC channel number
    switch(pinPortB)
    {
        case 5:  adcChannelNumber = 0; break;
        case 4:  adcChannelNumber = 2; break;
        case 3:  adcChannelNumber = 3; break;
        case 2:  adcChannelNumber = 1; break;
        case 1:  return 0;             break;    //No ADC channel on this pin
        case 0:  return 0;             break;    //No ADC channel on this pin
        default: return 0;             break;    //pin number out of range
    }

    //select the channel to read from (the lower 4 bits of ADMUX)
    //clear the lower 4 bits
    ADMUX &= ~( (1 << 0x00) | (1 << 0x01) | (1 << 0x02) | (1 << 0x03) );

    //now set them equal to "adcChannelNumber"
    ADMUX |= (adcChannelNumber & 0x03);  //only take the two lower bits of "adcChannelNumber", since the Attiny85 only have 4 ADC pins.

    // without a delay, we seem to read from the wrong channel
    delay(1);

    ADCSRA |= (1 << ADPS2) | (1 << ADPS1) | (1 << ADPS0); // Set ADC prescalar to 128, which is a 62.5KHz sample rate @ 8MHz
                                                          //If want to, we can crank it up to 200khz.

    // Enable ADC (only needs to be done once, but no side effects if we set this bit every time)
    ADCSRA |= (1 << ADEN);

    // start the conversion
    sbi(ADCSRA, ADSC);

    // ADSC is cleared when the conversion finishes
    while (bit_is_set(ADCSRA, ADSC));

    // we have to read ADCL first; doing so locks both ADCL
    // and ADCH until ADCH is read.  reading ADCL second would
    // cause the results of each conversion to be discarded,
    // as ADCL and ADCH would be locked when it completed.
    low = ADCL;
    high = ADCH;


    // combine the two bytes
    return (high << 8) | low;
}//end analogRead





uint8_t digitalRead(uint8_t pin)
{
    if( (PINB & (1<<pin)) > 0)
    {
        return HIGH;
    }
    else
    {
        return LOW;
    }
}//end digitalRead