// ****************************************************************************
// This code is distributed under the GNU Public License
//      which can be found at http://www.gnu.org/licenses/gpl.txt
//
// ****************************************************************************

#include "inttypes.h"
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/eeprom.h>
#include "stepperMotorDriver.h"
#include "PulseManager.h"
#include "utils.h"
#include "eepromDefines.h"
#include "AttinySoftwareSerial.h"
#include "config.h"



// This is the type for our control mode.
// The control mode gets written and retrived from the EEPROM.
// The underlining byte values are selected to be non-zero and have a
// hamming distance larger than two.
enum controlMode_t
{
    INVALID_CONTROL_MODE = 2,
    CONTINUOUS_ROTATION = 4,
    ABSOLUTE_POSITION = 8
};




//Forward Declearations
StepperMotor::direction_t pulseLengthToDirection(uint16_t pulseLength, uint16_t middleValue);
uint16_t                  pulseLengthToRPM(uint16_t pulseLength, uint16_t pulseMin, uint16_t pulseMid, uint16_t pulseMax, uint16_t rpmMin, uint16_t rpmMax);
uint16_t                  pulseLengthToPosition(uint16_t pulseLength, uint16_t min, uint16_t max);
controlMode_t             retriveControlModeFromEeprom();
void                      storeControlModeToEeprom(controlMode_t ctrlModeToStore);
/*
// Test out the Pulse Manager
int main()
{

    //set led pin as output
    DDRB |= (1 << PB3);

    //blink debug led to show that we are alive
    PORTB |= (1<<PB3);//on
    delay(2000);
    PORTB &= ~(1<<PB3);//off
    delay(1000);


    //Starts up the pulse manager, which measuress pulse widths on pin PB0
    PulseManager thePulseManager;



    DDRB |= (1 << 1); // Set serial TX pin as output

    AttinySoftwareSerial mySerial(0);
    mySerial.println( "ready for action" );

    uint16_t pulseLengthReading = 0;

    while(1)
    {


        if(thePulseManager.available() == true)
        {
            pulseLengthReading = thePulseManager.getPulseLength();
            mySerial.println( static_cast<uint32_t>(pulseLengthReading) );
        }
        else if( thePulseManager.wasModeChangePulseDetected() == true )
        {
            mySerial.println("*** MODE CHANGE PULSE DETECTED ***");
        }
        else
        {
            //do nothing
        }
    }

    return 0;
}
*/






///*
int main()
{



    //Set the stepping method based on whats defined in the config.h file.
    #ifdef USE_HALF_STEPS
        StepperMotor::stepMethod_t stepMethod = StepperMotor::HALF_STEP_MODE;
    #else
        StepperMotor::stepMethod_t stepMethod = StepperMotor::FULL_STEP_MODE;
    #endif



    // Create the pulse manager, which measures pulse widths on pin PB0.
    PulseManager thePulseManager;

    //Create a stepper that uses pins PB1 through PB4.
    StepperMotor myStepper(1,2,3,4, STEPS_PER_REVOLUTION_OF_MOTOR, stepMethod);

    //Set the stepper's move speed during absolute positioning mode to the speed
    // defined in the file config.h
    myStepper.defineMoveSpeedInAbsMode(ABSOLUTE_POSITION_MOVE_SPEED);


    controlMode_t controlMode        = CONTINUOUS_ROTATION;
    uint16_t      pulseLengthReading = 0;
    bool          updateStepper      = false;
    uint16_t      minSpeedInRpm      = MINIMUM_RPM;
    uint16_t      maxSpeedInRpm      = MAXIMUM_RPM;


    //Check if the user's desired min and max speed is within the supported range.
    if( minSpeedInRpm < myStepper.getMinSupportedRpm() )
    {
        minSpeedInRpm = myStepper.getMinSupportedRpm();
    }
    else
    {
       // The desired minimal speed is within the capability of the stepper driver.
       // Do nothing.
    }

    //Check if the user's desired min and max speed is within the supported range.
    if( maxSpeedInRpm > myStepper.getMaxSupportedRpm() )
    {
        maxSpeedInRpm = myStepper.getMaxSupportedRpm();
    }
    else
    {
       // The desired maximum speed is within the capability of the stepper driver.
       // Do nothing.
    }


    ////SERIAL DEBUG//
    //DDRB |= (1 << 1); // Set serial TX pin as output
    //
    //AttinySoftwareSerial mySerial(0);
    //mySerial.println( "ready for action" );
    ////END SERIAL DEBUG//

    //define what length input signal we are going to handle
    uint16_t minInputPulse = 1000;
    uint16_t centerInputPulse = 1500;
    uint16_t maxInputPulse = 2000;



    //Retrive the control mode from the EEPROM.
    controlMode = retriveControlModeFromEeprom();

    if( controlMode == INVALID_CONTROL_MODE)
    {
        controlMode = CONTINUOUS_ROTATION;
    }
    else
    {
        //We retrived a valid control mode from the EEPROM.
        // Do nothing else.
    }


    //if on start up our control mode is ABSOLUTE_POSITION
    if( controlMode == ABSOLUTE_POSITION)
    {
        //then we need to read in a pulse so we know our initial position

        //Eat up the first 20 pulses in order to prime the low pass filter and
        // wait for the input signal to stablize.
        for(int i = 0; i < 20; i++)
        {
            //busy wait for a new pulse measurement to be available
            while( thePulseManager.available() == false ){}
            //get the pulse length and throw it out.
            (void)thePulseManager.getPulseLength();
        }

        //busy wait for a new pulse measurement to be available
        while( thePulseManager.available() == false ){}
        //read the pulse length
        pulseLengthReading = thePulseManager.getPulseLength();
        pulseLengthReading = constrain(pulseLengthReading, minInputPulse, maxInputPulse);
        //convert pulse length into position
        uint16_t newStepperPos = pulseLengthToPosition(pulseLengthReading, minInputPulse, maxInputPulse);
        myStepper.defineCurrentPositionAsDegrees(newStepperPos);

    }
    else
    {
        //the control mode is CONTINUOUS_ROTATION
        //nothing special needs to be done here
    }



    ////*** DEBUG START ***
    //// slow sweep of all stepper positions
    //while(1)
    //{
    //    for(uint16_t i = 0; i < ABSOLUTE_POSITION_RANGE_IN_DEGREES; i++)
    //    {
    //        myStepper.gotoPositionDegrees(i);
    //        delay(100);
    //    }
    //
    //    for(uint16_t i = ABSOLUTE_POSITION_RANGE_IN_DEGREES; i > 0 ; i--)
    //    {
    //        myStepper.gotoPositionDegrees(i);
    //        delay(100);
    //    }
    //}
    ////*** DEBUG END ***



    ////*** DEBUG START ***
    //// Test out the goto position function
    //while(1)
    //{
    //    myStepper.gotoPositionDegrees(0);
    //    delay(4000);
    //    myStepper.gotoPositionDegrees(ABSOLUTE_POSITION_RANGE_IN_DEGREES);
    //    delay(4000);
    //}
    ////*** DEBUG END ***



    ////*** DEBUG START ***
    //
    //myStepper.setDirection(StepperMotor::FORWARD);
    ////myStepper.setSpeedRPM(10);
    //for(uint8_t i = 0; i < 100; i++)
    //{
    //    myStepper.setSpeedRPM(i);
    //    delay(100);
    //}
    //
    //while(1){};
    ////*** DEBUG END ***




    //loop forever
    while(1)
    {

        //==========================================================
        // Check if we should change operating mode.
        //==========================================================
        if( thePulseManager.wasModeChangePulseDetected() == true )
        {
            //toggle control mode
            if(controlMode == CONTINUOUS_ROTATION)
            {
                controlMode = ABSOLUTE_POSITION;
                //save the control mode to EEPROM
                storeControlModeToEeprom(controlMode);
                //stop the stepper from spinning
                myStepper.setSpeedRPM(0);
                //wait 200ms for it to come to a stop
                delay(200);
                //set its current position as the halfway point in the range of motion

                //myStepper.defineCurrentPositionAsDegrees(ABSOLUTE_POSITION_RANGE_IN_DEGREES/2);

                uint16_t newStepperPos = pulseLengthToPosition(pulseLengthReading, minInputPulse, maxInputPulse);
                myStepper.defineCurrentPositionAsDegrees(newStepperPos);
            }
            else if(controlMode == ABSOLUTE_POSITION)
            {
                controlMode = CONTINUOUS_ROTATION;
                //save the control mode to EEPROM
                storeControlModeToEeprom(controlMode);
            }
            else
            {
                //Invalid control mode. Should never get here. Do nothing.
            }
        }
        else
        {
            //The Mode Change Pulse was not detected. Don't toggle control mode.
        }


        //==========================================================
        // Check if input pulses have been missing for some time
        //==========================================================
        if( thePulseManager.wasSignalLossDetected() == true )
        {
           //yes it was. Remove power from our stepper motor.
           myStepper.unpower();
        }
        else
        {
            //Signal loss event was not detected. Do nothing.
        }


        //==========================================================
        // Check if a new pulse measurement is available.
        //==========================================================
        if( thePulseManager.available() == true )
        {
            //read the pulse length
            pulseLengthReading = thePulseManager.getPulseLength();
            pulseLengthReading = constrain(pulseLengthReading, minInputPulse, maxInputPulse);
            updateStepper = true;
        }
        else
        {
            //No new pulse is available. Do nothing.
        }


        //==========================================================
        // If we got new input, update the stepper speed, or
        // position, depending on our operating mode.
        //==========================================================
        if(updateStepper && controlMode == CONTINUOUS_ROTATION)
        {
            //convert pulse length into direction and speed
            StepperMotor::direction_t newStepperDir = pulseLengthToDirection(pulseLengthReading,
                                                                             centerInputPulse);

            uint16_t newStepperRPM = pulseLengthToRPM(pulseLengthReading, minInputPulse, centerInputPulse,
                                                        maxInputPulse, minSpeedInRpm, maxSpeedInRpm);

            //configure the stepper to spin in the new direction and speed
            myStepper.setDirection(newStepperDir);
            myStepper.setSpeedRPM(newStepperRPM);

            //clear the update flag
            updateStepper = false;
        }
        else if(updateStepper && controlMode == ABSOLUTE_POSITION)
        {
            //convert pulse length into position
            uint16_t newStepperPos = pulseLengthToPosition(pulseLengthReading, minInputPulse, maxInputPulse);
            myStepper.gotoPositionDegrees(newStepperPos);

            //clear the update flag
            updateStepper = false;
        }
        else
        {
            //We did not get any new input and therefor don't need to update the stepper
            // or we are somehow in some invalid control mode (which should never happen)
            //in either case, do nothing.
        }


    }//end while(1)



    return 0;
}//end main
//*/







//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
//
// INPUT:       Nothing
//
// RETURNS:
//=============================================================================
StepperMotor::direction_t pulseLengthToDirection(uint16_t pulseLength, uint16_t middleValue)
{
    if( pulseLength < middleValue )
    {
        return StepperMotor::BACKWARD;
    }
    else
    {
        return StepperMotor::FORWARD;
    }
}//end convertPulseLengthToDirection




//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
//
// INPUT:       Nothing
//
// RETURNS:
//=============================================================================
uint16_t pulseLengthToRPM(uint16_t pulseLength, uint16_t pulseMin, uint16_t pulseMid,
                          uint16_t pulseMax, uint16_t rpmMin, uint16_t rpmMax)
{
    uint16_t speedInRPM = 0;
    uint16_t deadZone = 16;

    //Check if the input pulse length is within the center deadzone.
    if( (pulseLength <= pulseMid+deadZone) && (pulseLength >= pulseMid-deadZone))
    {
        //It is within the deadzone. Set the speed to zero.
        speedInRPM = 0;
    }
    else
    {
        //The pulse length is not within the center dead zone. Continue to calculate
        //the speed in RPM.

        //Convert the pulse length into speed in Rotations Per Minute.
        if( pulseLength < pulseMid )
        {
            speedInRPM = map(pulseLength, (pulseMid-deadZone), pulseMin, rpmMin, rpmMax);
        }
        else
        {
            speedInRPM = map(pulseLength, (pulseMid+deadZone), pulseMax, rpmMin, rpmMax);
        }
    }

    return speedInRPM;

}//end convertPulseLengthToRPM




//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
//
// INPUT:       Nothing
//
// RETURNS:
//=============================================================================
uint16_t pulseLengthToPosition(uint16_t pulseLength, uint16_t min, uint16_t max)
{
    uint16_t minPos = 0;
    uint16_t maxPos = ABSOLUTE_POSITION_RANGE_IN_DEGREES;  //setting defined in config.h
    uint16_t positionToReturn = 0;

    positionToReturn = static_cast<uint16_t>( map(pulseLength, min, max, minPos, maxPos) );

    return positionToReturn;
}//end convertPulseLengthToPosition






//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
//
// INPUT:       Nothing
//
// RETURNS:
//=============================================================================
controlMode_t retriveControlModeFromEeprom()
{
    uint8_t ctrlModeFromEeprom = 0;
    ctrlModeFromEeprom = eeprom_read_byte((uint8_t*)CONTROL_MODE_EEPROM_ADDRESS);
    //CONTROL_MODE_EEPROM_ADDRESS is defined in the file eepromDefines.h

    switch( ctrlModeFromEeprom )
    {
    case CONTINUOUS_ROTATION:
        return CONTINUOUS_ROTATION;
        break;

    case ABSOLUTE_POSITION:
        return ABSOLUTE_POSITION;
        break;

    default:
        return INVALID_CONTROL_MODE;
        break;
    }
}//end retriveControlModeFromEeprom




//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
//
// INPUT:       Nothing
//
// RETURNS:
//=============================================================================
void storeControlModeToEeprom(controlMode_t ctrlModeToStore)
{
    eeprom_write_byte ((uint8_t*)CONTROL_MODE_EEPROM_ADDRESS, ctrlModeToStore);
}//end storeControlModeToEeprom