// ****************************************************************************
// This code is distributed under the GNU Public License
//      which can be found at http://www.gnu.org/licenses/gpl.txt
//
// ****************************************************************************

#ifndef STEPPERMOTORDRIVER_H_HEADERGUARD
#define STEPPERMOTORDRIVER_H_HEADERGUARD

#include "inttypes.h"


//This is the driver for a stepper motor
//Current limitations:
// - works only on port B
// - only up to one instance of the class is support (it can only control 1 stepper)


//Hardware resources used:
// Timer0


class StepperMotor
{

public:

    enum direction_t
    {
        UNDEFINED_DIRECTION = 0,
        FORWARD,
        BACKWARD
    };

    enum stepMethod_t
    {
        FULL_STEP_MODE = 0,
        HALF_STEP_MODE
    };

    //Constructor
    StepperMotor(uint8_t pinA, uint8_t pinB, uint8_t pinC, uint8_t pinD,
                 uint16_t newStepsPerRevOfMotor, stepMethod_t stepMethod);

    //TODO: create a destructor

    //Set the spin direction that will be used when speed is set using setSpeedRPM
    void setDirection(direction_t newDirection);

    //Makes motor continuously spin at the specified Rotations Per Minute
    void setSpeedRPM(uint16_t newSpeedInRPM);

    //Tells the motor to go to a specific position and stay there
    void gotoPosition(uint16_t newPosition);

    //Tells the motor to go to a specific position (in degrees) and stay there
    void gotoPositionDegrees(uint16_t newPositionInDegrees);

    //Defines the current position
    void defineCurrentPositionAs(uint16_t newPosition);

    //Defines the current position in degrees
    void defineCurrentPositionAsDegrees(uint16_t newPositionInDegrees);

    //Defines what speed to move at when in absolute positioning mode
    void defineMoveSpeedInAbsMode( uint16_t newSpeedInRpm);

    //Removes power from the motor, allowing it to spin freely.
    void unpower();

    //Returns the minimum supported RPM
    uint16_t getMinSupportedRpm();

    //Returns the maximum supported RPM
    uint16_t getMaxSupportedRpm();


    //Timer compare match handler. Should only be called when the compare match interrupt fires.
    static void timerCompareMatchISR();


private:

    enum operatingMode_t
    {
        CONTINUOUS = 1,
        ABSOLUTE,
        STATIONARY,
        UNPOWERED
    };

    //private functions
    void setStepInterval(uint32_t newIntervalInUs);
    void stepForward();
    void stepBackward();
    void calcRpmLimits();
    uint16_t degreesToSteps(uint16_t positionInDegrees);


    //private variables
             uint8_t         arrayOfMasks[8];
             uint8_t         arrayOfMasksLength;    //Number of elements in the arrayOfMasks array.
    volatile int8_t          stepSequence;
    volatile uint8_t         currentActiveMask;
             direction_t     stepDirection;
    volatile uint16_t        myPosition;
             uint16_t        targetPosition;
             uint16_t        moveSpeedDuringAbsMode;  //Speed to move at when in ABSOLUTE operating mode
             operatingMode_t operatingMode;
             uint8_t         myOutputCompareValue;    //used when updating the step interval
             uint8_t         myPrescalerMask;         //used when updating the step interval
             uint16_t        stepsPerRev;             //numbers of steps needed to make one revolution
             uint16_t        myMinSupportedRpm;       //minimum RPM this driver can do based on the motor's steps per rev and the stepping mode.
             uint16_t        myMaxSupportedRpm;       //maximum RPM this driver can do based on the motor's steps per rev and the stepping mode.


    static   StepperMotor* myThis;    //pointer to the (one and only) instance of this class.

};//end StepperMotor



#endif //STEPPERMOTORDRIVER_H_HEADERGUARD