// ****************************************************************************
// This code is distributed under the GNU Public License
//      which can be found at http://www.gnu.org/licenses/gpl.txt
//
// ****************************************************************************

//Stepper motor driver
// interrupt driven
// Supported time between steps: 0 to 32,768 microseconds + a couple microseconds of overhead.

//Hardware resources used:
// Timer0




#include "stepperMotorDriver.h"
#include <avr/io.h>
#include <avr/interrupt.h>
#include <util/atomic.h>

//static member variable initialization
StepperMotor* StepperMotor::myThis = 0;



//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
//
// INPUT:       Nothing
//
// RETURNS:
//=============================================================================
StepperMotor::StepperMotor(uint8_t pinA, uint8_t pinB, uint8_t pinC, uint8_t pinD,
                           uint16_t newStepsPerRevOfMotor, stepMethod_t stepMethod)
{
    //Create the array of bit masks based on the GPIO pins to use and the desired stepMethod.
    if(stepMethod == HALF_STEP_MODE)
    {
        //Set up the array with half steps.
        arrayOfMasks[0] =   (1<<pinA);
        arrayOfMasks[1] = ( (1<<pinA) | (1<<pinB) );
        arrayOfMasks[2] =   (1<<pinB);
        arrayOfMasks[3] = ( (1<<pinB) | (1<<pinC) );
        arrayOfMasks[4] =   (1<<pinC);
        arrayOfMasks[5] = ( (1<<pinC) | (1<<pinD) );
        arrayOfMasks[6] =   (1<<pinD);
        arrayOfMasks[7] = ( (1<<pinD) | (1<<pinA) );
        arrayOfMasksLength = 8;
        //With half stepping, the total number of steps in one rotation is double the
        // number of physical steps the motor has.
        stepsPerRev = newStepsPerRevOfMotor * 2;
    }
    else
    {
        //Set up the array with regular (full) steps.
        arrayOfMasks[0] = (1 << pinA);
        arrayOfMasks[1] = (1 << pinB);
        arrayOfMasks[2] = (1 << pinC);
        arrayOfMasks[3] = (1 << pinD);
        arrayOfMasksLength = 4;
        stepsPerRev = newStepsPerRevOfMotor;
    }

    //Set class variables to initial values.
    stepSequence         = 0;
    currentActiveMask    = 0;
    stepDirection        = UNDEFINED_DIRECTION;
    myPosition           = 0;
    targetPosition       = 0;
    operatingMode        = UNPOWERED;
    myOutputCompareValue = 0xFF;    //just a large value
    myPrescalerMask      = 0;

    // Call this function to set the myMinSupportedRpm and myMaxSupportedRpm variables
    // to the proper values based on the stepsPerRev
    calcRpmLimits();

    //Save the address of this object instance to the static variable "myThis".
    myThis = this;

    //Set selected GPIO pins as output.
    DDRB |= (1 << pinA);
    DDRB |= (1 << pinB);
    DDRB |= (1 << pinC);
    DDRB |= (1 << pinD);

    //Set prescaler to 1024 (the largest prescaler for timer0).
    //Each timer tick will equal 128 microseconds.
    myPrescalerMask |=  (1<< CS02); //set
    myPrescalerMask &= ~(1<< CS01); //clear
    myPrescalerMask |=  (1<< CS00); //set

    // Set the prescaler to its initial value
    TCCR0B |= myPrescalerMask;

    // Set the output compare register to its initial value
    OCR0A = myOutputCompareValue;

    // Enable Output Compare Match A Interrupt
    TIMSK |= (1 << OCIE0A);

    // Enable global interrupts
    sei();

    // This stepper driver only supports a single stepper motor. While this class will allow you
    // to make multiple instances of it, only the last instance created will actual drive a motor.
    // This is enforced by having the myThis variable overwritten each time a new instance of
    // this class is created.
}//end constructor



//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
//
// INPUT:       Nothing
//
// RETURNS:
//=============================================================================
void StepperMotor::setDirection(direction_t newDirection)
{
    ATOMIC_BLOCK(ATOMIC_FORCEON)
    {
        stepDirection = newDirection;
    }
}//end setDirection



//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
//
// INPUT:       Nothing
//
// RETURNS:
//=============================================================================
void StepperMotor::setSpeedRPM(uint16_t newSpeedInRPM)
{
    uint32_t delayBetweenSteps = 0;

    //check if the speed is too slow.
    if(newSpeedInRPM < myMinSupportedRpm)
    {
        //The speed is too slow. So slow that it is unsupported by this driver.
        //Stop moving, by changing our operating mode to 'STATIONARY'
        operatingMode = STATIONARY;
        return;
    }
    //Check if the speed is too fast.
    else if(newSpeedInRPM > myMaxSupportedRpm)
    {
        //The requested speed is too fast. Cap it at the max speed.
        newSpeedInRPM = myMaxSupportedRpm;
    }
    else
    {
        //The requested speed is not too fast and not too slow.
        //Keep going to honor it.
    }


    //convert from Rotations Per Minute to amount of microseconds to wait between steps
    //using this formula:
    //delay in microseconds = 1 / (RPM * 200 / 60) * 1000000
    //stepper has 200 steps per rotation
    //there are 60 seconds in a minute
    //1000000 microseconds in a second

    delayBetweenSteps = 1000000ul / ((uint32_t)newSpeedInRPM * (uint32_t)stepsPerRev / 60ul);

    //update the step speed
    setStepInterval(delayBetweenSteps);

    //change our control mode to indicate we are now free spinning
    operatingMode = CONTINUOUS;

}//end setSpeedRPM


//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
//
// INPUT:       Nothing
//
// RETURNS:
//=============================================================================
void StepperMotor::gotoPosition(uint16_t newPosition)
{
    //set the step speed we are going to use in ABSOLUTE control mode
    setSpeedRPM(moveSpeedDuringAbsMode);

    //set our new target position
    targetPosition = newPosition;

    //change our control mode to indicate we are moving to absolute positions
    operatingMode = ABSOLUTE;
}//end gotoPosition


//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
//
// INPUT:       Nothing
//
// RETURNS:
//=============================================================================
void StepperMotor::gotoPositionDegrees(uint16_t newPositionInDegrees)
{
    uint16_t newPositionInSteps = degreesToSteps(newPositionInDegrees);
    gotoPosition(newPositionInSteps);
}//end gotoPositionDegrees



//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
//
// INPUT:       Nothing
//
// RETURNS:
//=============================================================================
void StepperMotor::defineCurrentPositionAs(uint16_t newPosition)
{
    myPosition = newPosition;
    targetPosition = myPosition;
}//defineCurrentPositionAs


//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
//
// INPUT:       Nothing
//
// RETURNS:
//=============================================================================
void StepperMotor::defineCurrentPositionAsDegrees(uint16_t newPositionInDegrees)
{
    uint16_t newPosition = degreesToSteps(newPositionInDegrees);
    defineCurrentPositionAs(newPosition);
}//end defineCurrentPositionAsDegrees



//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
//
// INPUT:       Nothing
//
// RETURNS:
//=============================================================================
void StepperMotor::defineMoveSpeedInAbsMode( uint16_t newSpeedInRpm )
{
    if(newSpeedInRpm >= myMinSupportedRpm && newSpeedInRpm <= myMaxSupportedRpm)
    {
        moveSpeedDuringAbsMode = newSpeedInRpm;
    }
    else
    {
        //The desired move speed is either too fast or too slow than what this driver supports.
        //Do nothing.
    }
}//end defineMoveSpeedInAbsMode

//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
//
// INPUT:       Nothing
//
// RETURNS:
//=============================================================================
void StepperMotor::unpower()
{
    //update the operating mode to indicate we are unpowered
    operatingMode = UNPOWERED;

    //clear the current active mask. This sets the pin low, removing power from the motor
    PORTB &= ~currentActiveMask;
}//end unpower


//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
//
// INPUT:       Nothing
//
// RETURNS:
//=============================================================================
uint16_t StepperMotor::getMinSupportedRpm()
{
    return myMinSupportedRpm;
}//end getMinSupportedRpm


//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
//
// INPUT:       Nothing
//
// RETURNS:
//=============================================================================
uint16_t StepperMotor::getMaxSupportedRpm()
{
    return myMaxSupportedRpm;
}//end getMaxSupportedRpm



//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
//
// INPUT:       Nothing
//
// RETURNS:
//=============================================================================
void StepperMotor::timerCompareMatchISR()
{
    if( myThis->operatingMode == CONTINUOUS )
    {
        if(myThis->stepDirection == FORWARD)
        {
            myThis->stepForward();
        }
        else if(myThis->stepDirection == BACKWARD)
        {
            myThis->stepBackward();
        }
        else
        {
            //the direction is undefined. do nothing.
        }
    }
    else if(myThis->operatingMode == ABSOLUTE)
    {
        if(myThis->myPosition < myThis->targetPosition)
        {
            myThis->stepForward();
        }
        else if(myThis->myPosition > myThis->targetPosition)
        {
            myThis->stepBackward();
        }
        else
        {
            //We are at the target position. Take no steps.
        }
    }
    else if(myThis->operatingMode == STATIONARY)
    {
        //we are stationary, take no steps.
    }
    else if(myThis->operatingMode == UNPOWERED)
    {
        //The motor is unpowered, take no steps.
    }
    else
    {
        //Invalid operating mode. Should never get here.
    }

    //Update the prescaler and the output compare register.
    //Technically, we should do this only when the step interval was updated
    //but updating it everytime has no side-effects, and makes the code a bit simpler.

    //Update the prescaler. This is a two step process.
    TCCR0B &= ~( (1<< CS02) | (1<< CS01) | (1<< CS00) );    //set the clock select to all zeros
    TCCR0B |= myThis->myPrescalerMask;                      //Apply the bit mask to set the 1's

    //Update the output compare register
    OCR0A = myThis->myOutputCompareValue;

    //clear the compare flag, since it might have been set again in the last few clock cycles
    TIFR = (1<<OCF0A);    //flag is cleared by writting a logic 1 to it, and logic 0 to all other bits

    //reset the counter to 0
    TCNT0  = 0;
}//end timerCompareMatchISR






//=============================================================================
// Private functions
//=============================================================================

//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
//
// INPUT:       Nothing
//
// RETURNS:
//=============================================================================
void StepperMotor::stepForward()
{
    stepSequence++;
    if(stepSequence > (arrayOfMasksLength-1)) stepSequence = 0;    //Wrap around.

    //load the new mask
    uint8_t newMask = arrayOfMasks[stepSequence];

    //clear the last mask. This sets the current active pin to low.
    PORTB &= ~currentActiveMask;

    //apply the new mask. This sets the next active pin high
    PORTB |= newMask;

    //save the new mask as the current active mask
    currentActiveMask = newMask;

    //update the stepper position
    myPosition++;
}//end stepForward


//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
//
// INPUT:       Nothing
//
// RETURNS:
//=============================================================================
void StepperMotor::stepBackward()
{
    stepSequence--;
    if(stepSequence < 0) stepSequence = (arrayOfMasksLength-1);    //Wrap around.

    //load the new mask
    uint8_t newMask = arrayOfMasks[stepSequence];

    //clear the last mask. This sets the current active pin to low.
    PORTB &= ~currentActiveMask;

    //apply the new mask. This sets the next active pin high
    PORTB |= newMask;

    //save the new mask as the current active mask
    currentActiveMask = newMask;

    //update the stepper position
    myPosition--;
}//end stepBackward


//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
//
// INPUT:       Nothing
//
// RETURNS:
//=============================================================================
void StepperMotor::setStepInterval(uint32_t newIntervalInUs)
{
    //FYI: TCCR0B = Timer/Counter0 Control Register B
    //Datasheet, Page 83

    //Calculate what our new prescaler and output compare value should be and store them for later use.
    //We don't update the prescaler register or the output compare register in this function because
    //that would cause the current high pulse to be extended or terminated too early. Instead we update
    //these registers at the start of the next step.

    uint8_t prescalerBuffer = 0;
    uint8_t outputCompareBuffer = 0;


    if( newIntervalInUs < 256)
    {
        //set prescaler to 8
        //each timer tick will equal 1 microsecond
        prescalerBuffer &= ~(1<< CS02); //clear
        prescalerBuffer |=  (1<< CS01); //set
        prescalerBuffer &= ~(1<< CS00); //clear

        outputCompareBuffer = newIntervalInUs;
    }
    else if( newIntervalInUs < 2048)
    {
        //set prescaler to 64
        //each timer tick will equal 8 microseconds
        prescalerBuffer &= ~(1<< CS02); //clear
        prescalerBuffer |=  (1<< CS01); //set
        prescalerBuffer |=  (1<< CS00); //set

        outputCompareBuffer = newIntervalInUs/8;
    }
    else if( newIntervalInUs < 8192)
    {
        //set prescaler to 256
        //each timer tick will equal 32 microseconds
        prescalerBuffer |=  (1<< CS02); //set
        prescalerBuffer &= ~(1<< CS01); //clear
        prescalerBuffer &= ~(1<< CS00); //clear

        outputCompareBuffer = newIntervalInUs/32;
    }
    else if( newIntervalInUs < 32768)
    {
        //set prescaler to 1024
        //each timer tick will equal 128 microseconds
        prescalerBuffer |=  (1<< CS02); //set
        prescalerBuffer &= ~(1<< CS01); //clear
        prescalerBuffer |=  (1<< CS00); //set

        outputCompareBuffer = newIntervalInUs/128;
    }
    else
    {
        //requested step interval too high, can't handle it.
    }


    //Update these two variables atomicly. It is possible to get an interrupt after updating the first one
    // but before updating the second.
    ATOMIC_BLOCK(ATOMIC_FORCEON)
    {
        myOutputCompareValue = outputCompareBuffer;
        myPrescalerMask = prescalerBuffer;
    }

}//end setStepInterval




//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
//
// INPUT:       Nothing
//
// RETURNS:
//=============================================================================
void StepperMotor::calcRpmLimits()
{
    //These are the limits this driver supports, in microseconds.
    uint16_t minTimeBetweenSteps = 300;
    uint16_t maxTimeBetweenSteps = 30000;

    // RPM = 1 / (timeBetweenSteps / 60 * stepsPerRev) * 1000000
    // where 60 is the amount of seconds in a minute
    // and 1000000 is the amount of microseconds in a second

    myMinSupportedRpm = 1000000ul / ((uint32_t)maxTimeBetweenSteps * (uint32_t)stepsPerRev / 60ul);
    myMaxSupportedRpm = 1000000ul / ((uint32_t)minTimeBetweenSteps * (uint32_t)stepsPerRev / 60ul);
}//end calcRpmLimits



//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
//
// INPUT:       Nothing
//
// RETURNS:
//=============================================================================
uint16_t StepperMotor::degreesToSteps(uint16_t positionInDegrees)
{
    //convert the position in degrees into the position in steps
    return ((uint32_t)positionInDegrees * (uint32_t)stepsPerRev) / 360ul;
}//end degreesToSteps




//=============================================================================
// None member functions
//=============================================================================

//=============================================================================
// FUNCTION:    Interrupt service routine for timer0 compare A match
//
// DESCRIPTION: AVR Libc provided function that is vectored into when the
//              timer0 compare A match interrupt fires.
//
// INPUT:       Nothing
//
// RETURNS:     Nothing
//=============================================================================
ISR(TIM0_COMPA_vect, ISR_NOBLOCK)
{

    StepperMotor::timerCompareMatchISR();

}//end ISR TIM0_COMPA_vect
