// ****************************************************************************
// This code is distributed under the GNU Public License
//      which can be found at http://www.gnu.org/licenses/gpl.txt
//
// ****************************************************************************


// Pulse Manager
//
// Measures pulse lengths between 512 and 2,560 microseconds in 8 microsecond increments.
// Detects if the pulse length lasts for at least 3 seconds.
// Detects signal loss. Which is defined as no pulses seen for 130,560 microsecond.
// All other pulse measurements are ignored.
//
//
// Hardware resources used:
//  Timer1
//  INT0 interrupt for state changes on pin PB0
//
//
// Requirements:
//  Clock frequency must be 8Mhz
//
//
// Theory of operation:
//  The pulse measurement is broken up into 3 segments.

#include"PulseManager.h"

#include <avr/io.h>
#include <avr/interrupt.h>
#include <utils.h>


volatile bool    PulseManager::newPulseMeasurementIsAvailable = false;
volatile uint8_t PulseManager::latestPulseLength = 0;
volatile bool    PulseManager::modeChangeConditionDetected = false;
volatile bool    PulseManager::signalLossConditionDetected = false;
volatile uint8_t PulseManager::overflowCounter = 0;
volatile PulseManager::state_t PulseManager::state = IDLE;
         PulseManager*         PulseManager::myThis = 0;

//=============================================================================
// Public functions
//=============================================================================

//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
// INPUT:       Nothing
//
// RETURNS:
//
//=============================================================================
PulseManager::PulseManager()
:myLowPassfilter(49)
{
    myThis = this;
    //config the pin change interrupt

    //enable the Pin Change Interrupt
    GIMSK |= (1<<PCIE);

    //update the pin change mask to allow an interrupt when pin PB0 changes state.
    PCMSK |= (1<<PCINT0);


    //config the timer

    switchToIdleMode();

    ////clear the pin change interrupt flag, which might be set
    // GIFR |= (1<<PCIF);

    sei(); // Enable global interrupts

}//end constructor





//=============================================================================
// FUNCTION:    bool available()
//
// DESCRIPTION: Determines if a new servo pulse measurement is available
//
// INPUT:       Nothing
//
// RETURNS:     TRUE if a new measurement is ready to be read
//              FALSE if no new measurement is available since the last read
//                      ( since the last call to getPulseLength() )
//=============================================================================
bool PulseManager::available()
{
    return newPulseMeasurementIsAvailable;
}//end available



//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
// INPUT:       Nothing
//
// RETURNS:
//
//=============================================================================
uint16_t PulseManager::getPulseLength()
{
    if(newPulseMeasurementIsAvailable == true)
    {
        //clear the flag
        newPulseMeasurementIsAvailable = false;
        //filter the new pulse measurement
        latestPulseLength = myLowPassfilter.input(latestPulseLength);
    }
    else
    {
        //The newest pulse measurement is already filtered.
        //Do nothing.
    }

    return map(latestPulseLength,0,255,512,2560);
}//end getPulseLength



//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
// INPUT:       Nothing
//
// RETURNS:
//
//=============================================================================
bool PulseManager::wasModeChangePulseDetected()
{
    if(modeChangeConditionDetected == true)
    {
        //since this is a successful read, clear the flag
        modeChangeConditionDetected = false;
        return true;
    }
    else
    {
        return false;
    }
}//end wasModeChangePulseDetected


//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
// INPUT:       Nothing
//
// RETURNS:
//
//=============================================================================
bool PulseManager::wasSignalLossDetected()
{
    if(signalLossConditionDetected == true)
    {
        //since this is a successful read, clear the flag
        signalLossConditionDetected = false;
        return true;
    }
    else
    {
        return false;
    }
}//end wasSignalLossDetected

















//=============================================================================
// Private functions
//=============================================================================

//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
// INPUT:       Nothing
//
// RETURNS:
//
//=============================================================================
void PulseManager::switchToIdleMode()
{
    //set prescaler to 4096, which makes each timer tick 512 microseconds
    // This means the timer will overflow after 130,560 microseconds
    TCCR1 |=  (1<<CS13); //set
    TCCR1 |=  (1<<CS12); //set
    TCCR1 &= ~(1<<CS11); //clear
    TCCR1 |=  (1<<CS10); //set

    //set the state variable
    state = IDLE;
}//end switchToIdleMode


//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
// INPUT:       Nothing
//
// RETURNS:
//
//=============================================================================
void PulseManager::switchToTimingServoPulsePreamble()
{
    //set prescaler to 16, which makes each timer tick 2 microseconds
    TCCR1 &= ~(1<<CS13); //clear
    TCCR1 |=  (1<<CS12); //set
    TCCR1 &= ~(1<<CS11); //clear
    TCCR1 |=  (1<<CS10); //set

    //set the state variable
    state = TIMING_SERVO_PREAMBLE;
}//end  switchToTimingServoPulsePreamble


//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
// INPUT:       Nothing
//
// RETURNS:
//
//=============================================================================
void PulseManager::switchToMeasuringServoPulse()
{
    //set prescaler to 64, which makes each timer tick 8 microseconds
    TCCR1 &= ~(1<<CS13); //clear
    TCCR1 |=  (1<<CS12); //set
    TCCR1 |=  (1<<CS11); //set
    TCCR1 |=  (1<<CS10); //set

    //set the state variable
    state = MEASURING_SERVO_PULSE;
}//end switchToMeasuringServoPulse


//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
// INPUT:       Nothing
//
// RETURNS:
//
//=============================================================================
void PulseManager::switchToMeasuringLongPulse()
{
    //set prescaler to 16384, which makes each timer tick 2048 microseconds
    TCCR1 |=  (1<<CS13); //set
    TCCR1 |=  (1<<CS12); //set
    TCCR1 |=  (1<<CS11); //set
    TCCR1 |=  (1<<CS10); //set

    //set the state variable
    state = MEASURING_LONG_PULSE;
}//end switchToMeasuringLongPulse


//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
// INPUT:       Nothing
//
// RETURNS:
//
//=============================================================================
void PulseManager::pinStateChangeISR()
{

    if(state == IDLE)
    {
        //read the state of the pin
        if( (PINB & (1<<PB0)) > 0)
        {
            //pin is high
            //At this point the pin has been low for a maybe a while and has gone high.
            //This is the start of the pulse we are going to measure.
            //start the timer that measures pulse length
            TCNT1 = 0; //set the timer to 0

            // Enable the Timer1 overflow interrupt
            TIMSK |= (1 << TOIE1);

            //change mode to timing the preamble
            myThis->switchToTimingServoPulsePreamble();

            //clear the overflow flag, since it might have been set while this interrupt was off
            TIFR = (1<<TOV1); //flag is cleared by writting a logic 1 to it, and logic 0 to all other bits
        }
        else
        {
            //pin is low
            //This case should never happen. Do nothing.
        }
    }


    else if(state == TIMING_SERVO_PREAMBLE)
    {
        //read the state of the pin
        if( (PINB & (1<<PB0)) > 0)
        {
            //pin is high
            //This case should never happen
        }
        else
        {
            //pin is low
            //The pin went low while we where still in the preamble period.
            //This pulse length is too short for us to count it as valid.
            //Drop this measurement on the floor and reset our variables to
            //be ready for the start of the next pulse.

            //Go back to idle
            myThis->switchToIdleMode();

            //set the timer to 0
            TCNT1 = 0;
        }
    }


    else if(state == MEASURING_SERVO_PULSE)
    {
        //read the state of the pin
        if( (PINB & (1<<PB0)) > 0)
        {
            //pin is high
            //This case should never happen, do nothing.
        }
        else
        {
            //pin is low
            //stop the timer and save the pulse length
            latestPulseLength = TCNT1;

            //set flag to indicate a new pulse measurement is ready
            newPulseMeasurementIsAvailable = true;

            //clear the flag since we just got a valid pulse measurement.
            //This flag might be set if this is the first valid pulse after a period of no pulses.
            signalLossConditionDetected = false;

            //change mode back to idle
            myThis->switchToIdleMode();

            //set the timer to 0
            TCNT1 = 0;
        }
    }


    else if(state == MEASURING_LONG_PULSE)
    {
        //read the state of the pin
        if( (PINB & (1<<PB0)) > 0)
        {
            //pin is high
            //This case should never happen, do nothing.
        }
        else
        {
            //pin is low
            //change mode back to idle
            myThis->switchToIdleMode();

            //set the timer to 0
            TCNT1 = 0;

            //reset the overflow counter, since we done measuring the long pulse
            overflowCounter = 0;
        }
    }


    else
    {
        //invalid state! Should never get here.
    }

    //clear the pin change interrupt flag, which might be set
    GIFR = (1<<PCIF);

}//end pinStateChangeISR


//=============================================================================
// FUNCTION:
//
// DESCRIPTION:
//
// INPUT:       Nothing
//
// RETURNS:
//
//=============================================================================
void PulseManager::timerOverflowISR()
{
    if(state == IDLE)
    {
        //the timer has over flowed which means we have not received a single pulse the entire time (130,560 microsecond)
        signalLossConditionDetected = true;
    }
    else if(state == TIMING_SERVO_PREAMBLE)
    {
        //This pulse lasted long enough to get through the preamble, go to the next part.
        myThis->switchToMeasuringServoPulse();
    }
    else if(state == MEASURING_SERVO_PULSE)
    {
        //ok, looks like this is going to be a long pulse, change measuring mode.
        myThis->switchToMeasuringLongPulse();
    }
    else if(state == MEASURING_LONG_PULSE)
    {
        //bump up the overflowCounter, without letting it roll over
        if(overflowCounter< 255) overflowCounter++;

        //detect if pulse length is at least 3 seconds long
        if(overflowCounter == 6) modeChangeConditionDetected = true;
    }
}//end timerOverflowISR




















//=============================================================================
// None-class function definitions
//=============================================================================

//=============================================================================
// FUNCTION:    Interrupt service routine for INT0
//
// DESCRIPTION: AVR Libc provided function that is vectored into when the
//              logic level on pin PB0 changes
//
// INPUT:       Nothing
//
// RETURNS:     Nothing
//=============================================================================
ISR(PCINT0_vect)
{
    PulseManager::pinStateChangeISR();

}//end ISR INT0_vect







//=============================================================================
// FUNCTION:    Interrupt service routine for timer1 overflows
//
// DESCRIPTION: AVR Libc provided function that is vectored into when the
//              timer1 overflow interrupt fires.
//
// INPUT:       Nothing
//
// RETURNS:     Nothing
//=============================================================================
ISR(TIM1_OVF_vect)
{
    PulseManager::timerOverflowISR();

}//end ISR TIM0_COMPA_vect













//don't really need a compare match interrupt
////=============================================================================
//// FUNCTION:    Interrupt service routine for timer1 compare A match
////
//// DESCRIPTION: AVR Libc provided function that is vectored into when the
////              timer1 compare A match interrupt fires.
////
//// INPUT:       Nothing
////
//// RETURNS:     Nothing
////=============================================================================
//ISR(TIM1_COMPA_vect)
//{
//    PulseManager::timerCompareMatchISR();
//}//end ISR TIM0_COMPA_vect
//
//void PulseManager::timerCompareMatchISR()
//{
//
//}//end timerCompareMatchISR


//// Disable the Timer1 overflow interrupt
//TIMSK &= ~(1 << TOIE1);